<?php
/**
 * Plugin Name: Phone Click Counter (tel: + cap IP)
 * Description: Compte les clics sur les liens téléphone (tel:) ayant la classe CSS "siteLogoDesktopPhoneNr", avec limite 10 clics/IP/jour + stats/jour + graphe admin.
 * Version: 1.1.0
 * Author: Votre Nom
 */

if (!defined('ABSPATH')) exit;

class Phone_Click_Counter {
    const CLASS_NAME = 'siteLogoDesktopPhoneNr';
    const OPTION_INSTALL_DATE = 'pcc_install_date';
    const TABLE_DAILY = 'pcc_daily_clicks';
    const TABLE_IP = 'pcc_ip_daily';
    const IP_DAILY_CAP = 10;

    public function __construct() {
        register_activation_hook(__FILE__, [$this, 'activate']);

        add_action('wp_enqueue_scripts', [$this, 'enqueue_front']);

        add_action('admin_menu', [$this, 'admin_menu']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);

        add_action('wp_ajax_nopriv_pcc_track', [$this, 'ajax_track']);
        add_action('wp_ajax_pcc_track', [$this, 'ajax_track']);
    }

    public function activate() {
        global $wpdb;
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');

        $charset_collate = $wpdb->get_charset_collate();
        $daily = $wpdb->prefix . self::TABLE_DAILY;
        $ip = $wpdb->prefix . self::TABLE_IP;

        // Table agrégée par jour
        $sql1 = "CREATE TABLE $daily (
            day DATE NOT NULL,
            clicks BIGINT UNSIGNED NOT NULL DEFAULT 0,
            PRIMARY KEY (day)
        ) $charset_collate;";

        // Table par IP/jour (on stocke un hash, pas l'IP en clair)
        $sql2 = "CREATE TABLE $ip (
            day DATE NOT NULL,
            ip_hash CHAR(64) NOT NULL,
            clicks SMALLINT UNSIGNED NOT NULL DEFAULT 0,
            PRIMARY KEY (day, ip_hash),
            KEY day_idx (day)
        ) $charset_collate;";

        dbDelta($sql1);
        dbDelta($sql2);

        if (!get_option(self::OPTION_INSTALL_DATE)) {
            update_option(self::OPTION_INSTALL_DATE, current_time('Y-m-d'));
        }
    }

    public function enqueue_front() {
        wp_enqueue_script(
            'pcc-front',
            plugins_url('assets/pcc.js', __FILE__),
            [],
            '1.1.0',
            true
        );

        wp_localize_script('pcc-front', 'PCC', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'action'  => 'pcc_track',
            'className' => self::CLASS_NAME,
            'nonce' => wp_create_nonce('pcc_track_nonce'),
        ]);
    }

    private function get_client_ip(): string {
        // On tente de récupérer l'IP réelle (attention: dépend de l’infra proxy)
        $candidates = [
            'HTTP_CF_CONNECTING_IP', // Cloudflare
            'HTTP_X_REAL_IP',
            'HTTP_X_FORWARDED_FOR',
            'REMOTE_ADDR',
        ];

        foreach ($candidates as $key) {
            if (empty($_SERVER[$key])) continue;

            $value = trim((string)$_SERVER[$key]);

            // X_FORWARDED_FOR peut contenir une liste "client, proxy1, proxy2"
            if ($key === 'HTTP_X_FORWARDED_FOR' && strpos($value, ',') !== false) {
                $parts = array_map('trim', explode(',', $value));
                $value = $parts[0] ?? $value;
            }

            if (filter_var($value, FILTER_VALIDATE_IP)) {
                return $value;
            }
        }

        return '0.0.0.0';
    }

    private function ip_to_hash(string $ip): string {
        // Hash salé via wp_hash (utilise les AUTH keys WP)
        // -> évite de stocker l'IP brute (meilleur RGPD)
        return hash('sha256', wp_hash($ip));
    }

    public function ajax_track() {
        $nonce = isset($_POST['nonce']) ? sanitize_text_field($_POST['nonce']) : '';
        if (!wp_verify_nonce($nonce, 'pcc_track_nonce')) {
            wp_send_json_error(['message' => 'Invalid nonce'], 403);
        }

        $href = isset($_POST['href']) ? (string)$_POST['href'] : '';
        $cls  = isset($_POST['cls']) ? sanitize_text_field($_POST['cls']) : '';

        if ($cls !== self::CLASS_NAME) {
            wp_send_json_error(['message' => 'Wrong class'], 400);
        }

        // On ne compte que les liens tel:
        $href_trim = trim($href);
        if (stripos($href_trim, 'tel:') !== 0) {
            wp_send_json_error(['message' => 'Not a tel: link'], 400);
        }

        $day = current_time('Y-m-d');

        $ip = $this->get_client_ip();
        $ip_hash = $this->ip_to_hash($ip);

        global $wpdb;
        $table_daily = $wpdb->prefix . self::TABLE_DAILY;
        $table_ip = $wpdb->prefix . self::TABLE_IP;

        // Lire le compteur IP/jour
        $current_ip_clicks = (int)$wpdb->get_var(
            $wpdb->prepare(
                "SELECT clicks FROM $table_ip WHERE day = %s AND ip_hash = %s",
                $day,
                $ip_hash
            )
        );

        // Cap atteint -> on ne compte pas, mais on renvoie OK pour ne pas perturber l’utilisateur
        if ($current_ip_clicks >= self::IP_DAILY_CAP) {
            wp_send_json_success([
                'day' => $day,
                'counted' => false,
                'reason' => 'ip_cap_reached',
                'cap' => self::IP_DAILY_CAP
            ]);
        }

        // Incrémenter la ligne IP/jour (UPSERT)
        $wpdb->query(
            $wpdb->prepare(
                "INSERT INTO $table_ip (day, ip_hash, clicks) VALUES (%s, %s, 1)
                 ON DUPLICATE KEY UPDATE clicks = clicks + 1",
                $day,
                $ip_hash
            )
        );

        // Incrémenter l'agrégat jour (UPSERT)
        $wpdb->query(
            $wpdb->prepare(
                "INSERT INTO $table_daily (day, clicks) VALUES (%s, 1)
                 ON DUPLICATE KEY UPDATE clicks = clicks + 1",
                $day
            )
        );

        wp_send_json_success([
            'day' => $day,
            'counted' => true,
            'cap' => self::IP_DAILY_CAP
        ]);
    }

    public function admin_menu() {
        add_menu_page(
            'Phone Clicks',
            'Phone Clicks',
            'manage_options',
            'pcc-phone-clicks',
            [$this, 'admin_page'],
            'dashicons-chart-bar',
            80
        );
    }

    public function enqueue_admin_assets($hook) {
        // Charge uniquement sur la page du plugin
        if ($hook !== 'toplevel_page_pcc-phone-clicks') return;

        // Chart.js via CDN (simple & efficace)
        wp_enqueue_script(
            'pcc-chartjs',
            'https://cdn.jsdelivr.net/npm/chart.js@4.4.1/dist/chart.umd.min.js',
            [],
            '4.4.1',
            true
        );

        wp_enqueue_script(
            'pcc-admin',
            plugins_url('assets/pcc-admin.js', __FILE__),
            ['pcc-chartjs'],
            '1.1.0',
            true
        );

        // Injecter les données pour le graphe
        $data = $this->get_daily_rows();
        wp_localize_script('pcc-admin', 'PCC_ADMIN', [
            'labels' => array_column($data, 'day'),
            'values' => array_map('intval', array_column($data, 'clicks')),
        ]);
    }

    private function get_daily_rows(): array {
        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_DAILY;
        $install_date = get_option(self::OPTION_INSTALL_DATE, current_time('Y-m-d'));

        $rows = $wpdb->get_results(
            $wpdb->prepare("SELECT day, clicks FROM $table WHERE day >= %s ORDER BY day ASC", $install_date),
            ARRAY_A
        );

        return is_array($rows) ? $rows : [];
    }

    public function admin_page() {
        if (!current_user_can('manage_options')) return;

        $install_date = get_option(self::OPTION_INSTALL_DATE, current_time('Y-m-d'));
        $rows = $this->get_daily_rows();

        $total = 0;
        foreach ($rows as $r) $total += (int)$r['clicks'];

        echo '<div class="wrap">';
        echo '<h1>Stats clics téléphone</h1>';
        echo '<p><strong>Classe suivie :</strong> <code>' . esc_html(self::CLASS_NAME) . '</code></p>';
        echo '<p><strong>Règle :</strong> uniquement <code>tel:</code> + limite <strong>' . esc_html(self::IP_DAILY_CAP) . ' clics/IP/jour</strong></p>';
        echo '<p><strong>Date d’installation :</strong> ' . esc_html($install_date) . '</p>';
        echo '<p><strong>Total clics comptabilisés :</strong> ' . esc_html($total) . '</p>';

        echo '<hr />';
        echo '<h2>Graphe (clics par jour)</h2>';
        echo '<div style="max-width: 1000px;"><canvas id="pccChart" height="120"></canvas></div>';

        echo '<hr />';
        echo '<h2>Détail (clics par jour)</h2>';

        if (empty($rows)) {
            echo '<p>Aucune donnée pour le moment.</p>';
        } else {
            echo '<table class="widefat striped" style="max-width:700px">';
            echo '<thead><tr><th>Date</th><th>Clics</th></tr></thead><tbody>';
            foreach ($rows as $r) {
                echo '<tr>';
                echo '<td>' . esc_html($r['day']) . '</td>';
                echo '<td>' . esc_html($r['clicks']) . '</td>';
                echo '</tr>';
            }
            echo '</tbody></table>';
        }

        echo '</div>';
    }
}

new Phone_Click_Counter();
